import streamlit as st
import yfinance as yf
import pandas as pd
import matplotlib.pyplot as plt
from sumy.parsers.plaintext import PlaintextParser
from sumy.nlp.tokenizers import Tokenizer
from sumy.summarizers.lsa import LsaSummarizer

st.set_page_config(layout="wide")
st.title("📈 Stock Dashboard + 📝 Text Summarizer")

# Session state for ticker
if "ticker" not in st.session_state:
    st.session_state.ticker = "AAPL"

# Sidebar Input
with st.sidebar:
    st.header("🔍 Stock Settings")
    ticker_input = st.text_input("Enter stock ticker:", value=st.session_state.ticker)
    period = st.selectbox("Select period", ["1mo", "3mo", "6mo", "1y", "2y", "5y"])
    fetch_btn = st.button("Fetch Data")

# Update session ticker
if fetch_btn:
    st.session_state.ticker = ticker_input.upper()

# Fetch and display stock data
ticker = st.session_state.ticker
data = yf.Ticker(ticker).history(period=period)

st.subheader(f"📊 Stock Price Chart for {ticker}")
if not data.empty:
    st.line_chart(data["Close"])
    st.metric("Current Price", f"${data['Close'][-1]:.2f}")
else:
    st.warning("No data available. Please check the ticker symbol.")

# Text summarization section
st.subheader("📝 Text Summarization")

text_input = st.text_area("Paste any stock-related article or news here:", height=200)
num_sentences = st.slider("Number of sentences in summary", 1, 5, 2)

def summarize_text(text, num_sentences):
    parser = PlaintextParser.from_string(text, Tokenizer("english"))
    summarizer = LsaSummarizer()
    summary = summarizer(parser.document, num_sentences)
    return " ".join([str(sentence) for sentence in summary])

if st.button("Summarize"):
    if text_input.strip():
        summary = summarize_text(text_input, num_sentences)
        st.success("Summary:")
        st.write(summary)
    else:
        st.warning("Please paste some text.")
