import streamlit as st
import yfinance as yf
import pandas as pd
from sumy.parsers.plaintext import PlaintextParser
from sumy.nlp.tokenizers import Tokenizer
from sumy.summarizers.lsa import LsaSummarizer

# Page configuration
st.set_page_config(page_title="Stock Dashboard & Summarizer", layout="wide")

# App Title and Description
st.title("📈 Stock Price Dashboard & 📝 Summarizer")
st.markdown("""
Welcome to the **Stock Analysis App** built with Streamlit.
- Search and visualize **stock price trends**
- Paste stock-related news or description for **automatic summarization**
""")

# User Input Section
st.sidebar.header("User Input")
stock_symbols = st.sidebar.multiselect(
    "Select Stock Symbols",
    ["AAPL", "MSFT", "GOOGL", "AMZN", "META", "TSLA", "NVDA", "JPM", "V", "WMT", "JSWSTEEL.NS"],
    default=["AAPL"]
)
days = st.sidebar.slider("Select number of past days", min_value=5, max_value=365, value=30)

# Function to fetch stock data
@st.cache_data
def fetch_stock_data(symbol, days):
    try:
        data = yf.download(symbol, period=f"{days}d")
        return data
    except Exception as e:
        return None

def display_stock_chart(stock_data, symbol, use_full_width=False):
    if use_full_width:
        st.subheader(f"{symbol} Stock Price")
        st.line_chart(stock_data['Close'], use_container_width=True)
        latest_price = float(stock_data['Close'].iloc[-1])
        st.metric(f"{symbol} Latest Close Price", f"${latest_price:.2f}")
    else:
        with col1 if idx % 2 == 0 else col2:
            st.subheader(f"{symbol} Stock Price")
            st.line_chart(stock_data['Close'], use_container_width=True)
            latest_price = float(stock_data['Close'].iloc[-1])
            st.metric(f"{symbol} Latest Close Price", f"${latest_price:.2f}")

# Display stock chart
st.subheader(f"📊 Stock Price Charts")
if st.sidebar.button("Load Stock Data"):
    total_stocks = len(stock_symbols)
    
    # For multiple stocks, create two columns
    if total_stocks > 1:
        col1, col2 = st.columns(2)
    
    for idx, symbol in enumerate(stock_symbols):
        try:
            stock_data = fetch_stock_data(symbol, days)
            if stock_data is not None and not stock_data.empty:
                # Use full width for single stock or last stock in odd number
                use_full_width = total_stocks == 1 or (idx == total_stocks - 1 and total_stocks % 2 == 1)
                display_stock_chart(stock_data, symbol, use_full_width)
            else:
                st.warning(f"No data found for symbol {symbol}.")
        except Exception as e:
            st.error(f"Error fetching data for {symbol}: {str(e)}")

# Text Summarization Section
st.markdown("---")
st.subheader("📝 Stock News or Description Summarizer")
text_input = st.text_area("Paste any stock-related text here to summarize")

# Function to summarize text
def summarize_text(text, sentence_count=2):
    parser = PlaintextParser.from_string(text, Tokenizer("english"))
    summarizer = LsaSummarizer()
    summary = summarizer(parser.document, sentence_count)
    return " ".join(str(sentence) for sentence in summary)

# Button to trigger summarization
if st.button("Generate Summary"):
    if text_input.strip():
        summary = summarize_text(text_input)
        st.success("**Summary:**")
        st.write(summary)
    else:
        st.warning("Please enter some text first.")

# Footer
st.markdown("---")
st.markdown("Built with [Streamlit](https://streamlit.io/) | By Mitul Lakkad")