SECRET_KEY = "supersecretkey123"
ALGORITHM = "HS256"

from fastapi import FastAPI, Depends, WebSocket, WebSocketDisconnect
from sqlalchemy.orm import Session  # Import the Session class
from database import engine, Base, SessionLocal, get_db
import models, schemas, crud
from auth import router as auth_router
from auth import get_password_hash
from websocket_manager import connect_client, disconnect_client, notify_all
from fastapi.security import OAuth2PasswordBearer
from jose import JWTError, jwt
from fastapi import HTTPException

app = FastAPI()
Base.metadata.create_all(bind=engine)
app.include_router(auth_router)

@app.post("/signup", response_model=schemas.User)
def signup(user: schemas.UserCreate, db: Session = Depends(get_db)):
    existing_user = crud.get_user_by_username(db, user.username)
    if existing_user:
        raise HTTPException(status_code=400, detail="Username already registered")
    hashed_password = get_password_hash(user.password)
    return crud.create_user(db, user, hashed_password)

oauth2_scheme = OAuth2PasswordBearer(tokenUrl="login")

def get_current_user(token: str = Depends(oauth2_scheme), db: Session = Depends(get_db)):
    credentials_exception = HTTPException(
        status_code=401,
        detail="Could not validate credentials",
        headers={"WWW-Authenticate": "Bearer"},
    )
    try:
        payload = jwt.decode(token, SECRET_KEY, algorithms=[ALGORITHM])
        username: str = payload.get("sub")
        if username is None:
            raise credentials_exception
    except JWTError:
        raise credentials_exception
    user = crud.get_user_by_username(db, username)
    if user is None:
        raise credentials_exception
    return user

# @app.get("/")
# def root():
#     return {"message": "Welcome to the Task Manager 🚀"}


@app.get("/todos/", response_model=list[schemas.Todo])
def get_user_todos(
    db: Session = Depends(get_db),
    current_user: models.User = Depends(get_current_user)
):
    return crud.get_todos_by_user(db, user_id=current_user.id)

@app.post("/todos", response_model=schemas.Todo)
async def create_todo(
    todo: schemas.TodoCreate,
    db: Session = Depends(get_db),  # This retrieves the database session
    current_user: models.User = Depends(get_current_user)  # This retrieves the current user
):
    new_todo = crud.create_todo(db=db, todo=todo, user_id=current_user.id)
    await notify_all({"event": "new_todo", "todo": new_todo})
    return new_todo

@app.put("/todos/{todo_id}", response_model=schemas.Todo)
async def update_todo(
    todo_id: int,
    todo: schemas.TodoCreate,
    db: Session = Depends(get_db),
    current_user: models.User = Depends(get_current_user)
):
    updated_todo = crud.update_todo(db=db, todo_id=todo_id, todo=todo)
    await notify_all({"event": "update_todo", "todo": updated_todo})
    return updated_todo

@app.delete("/todos/{todo_id}", response_model=dict)
async def delete_todo(
    todo_id: int,
    db: Session = Depends(get_db),
    current_user: models.User = Depends(get_current_user)
):
    return crud.delete_todo(db=db, todo_id=todo_id)

@app.websocket("/ws")
async def websocket_endpoint(websocket: WebSocket):
    await connect_client(websocket)
    try:
        while True:
            await websocket.receive_text()  # Keep the connection open
    except WebSocketDisconnect:
        await disconnect_client(websocket)