from sqlalchemy.orm import Session
from sqlalchemy.exc import SQLAlchemyError
from sqlalchemy.future import select
from fastapi import status
from datetime import datetime, timezone
from app.global_constants import GlobalConstants
from app.models.user_model import User
from app.models.todo_model import (
    ToDo,
    ToDoCreateSchema,
    ToDoUpdateSchema,
    ToDoResponseSchema,
)

class ToDoService:
    def __init__(self, db_session: Session):
        self.db_session = db_session


    def create_todo(self, todo_data: ToDoCreateSchema):
        try:
            
            user_exists = self.db_session.execute(
                select(User).where(User.id == todo_data.created_by)
            ).scalars().first()
            if not user_exists:
                return GlobalConstants.return_api_response(
                    message=f"User with ID {todo_data.created_by} not found.",
                    result=[],
                    status_code=status.HTTP_400_BAD_REQUEST
                )
                
            new_todo = ToDo(
                title=todo_data.title,
                description=todo_data.description,
                is_completed=False,
                created_at=datetime.now(timezone.utc),
                updated_at=datetime.now(timezone.utc),
                created_by=todo_data.created_by,
                updated_by=todo_data.created_by,
            )
            self.db_session.add(new_todo)
            self.db_session.commit()
            self.db_session.refresh(new_todo)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.accepted,
                result=[ToDoResponseSchema(
                    id=new_todo.id,
                    title=new_todo.title,
                    description=new_todo.description,
                    is_completed=new_todo.is_completed,
                    created_at=new_todo.created_at.isoformat() if new_todo.created_at else None,
                    updated_at=new_todo.updated_at.isoformat() if new_todo.updated_at else None,
                    created_by=new_todo.created_by,
                    updated_by=new_todo.updated_by,
                )],
                status_code=status.HTTP_201_CREATED
            )
        except SQLAlchemyError as e:
            self.db_session.rollback()
            print(f"{GlobalConstants.api_response_messages.error_in} create_todo service: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.db_error,
                result=None,
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR
            )

    def get_all_todos(self):
        try:
            result = self.db_session.execute(select(ToDo))
            todos = result.scalars().all()
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.success,
                result=[ToDoResponseSchema(
                    id=todo.id,
                    title=todo.title,
                    description=todo.description,
                    is_completed=todo.is_completed,
                    created_at=todo.created_at.isoformat() if todo.created_at else None,
                    updated_at=todo.updated_at.isoformat() if todo.updated_at else None,
                    created_by=todo.created_by,
                    updated_by=todo.updated_by,
                ) for todo in todos],
                status_code=status.HTTP_200_OK
            )
        except SQLAlchemyError as e:
            print(f"{GlobalConstants.api_response_messages.error_in} get_all_todos service: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.db_error,
                result=None,
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR
            )

    def get_todo_by_id(self, todo_id: str):
        try:
            result = self.db_session.execute(
                select(ToDo).where(ToDo.id == todo_id)
            )
            todo = result.scalars().first()
            if not todo:
                return GlobalConstants.return_api_response(
                    message=f"ToDo with ID {todo_id} not found.",
                    result=[],
                    status_code=status.HTTP_400_BAD_REQUEST
                )
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.success,
                result=[ToDoResponseSchema(
                    id=todo.id,
                    title=todo.title,
                    description=todo.description,
                    is_completed=todo.is_completed,
                    created_at=todo.created_at.isoformat() if todo.created_at else None,
                    updated_at=todo.updated_at.isoformat() if todo.updated_at else None,
                    created_by=todo.created_by,
                    updated_by=todo.updated_by,
                )],
                status_code=status.HTTP_200_OK
            )
        except SQLAlchemyError as e:
            print(f"{GlobalConstants.api_response_messages.error_in} get_todo_by_id service: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.db_error,
                result=None,
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR
            )

    def update_todo(self, todo_update: ToDoUpdateSchema):
        try:
            result = self.db_session.execute(
                select(ToDo).where(ToDo.id == todo_update.todo_id)
            )
            todo = result.scalars().first()
            if not todo:
                return GlobalConstants.return_api_response(
                    message=f"ToDo with ID {todo_update.todo_id} not found.",
                    result=[],
                    status_code=status.HTTP_400_BAD_REQUEST
                )
            if todo_update.title is not None:
                todo.title = todo_update.title
            if todo_update.description is not None:
                todo.description = todo_update.description
            if todo_update.is_completed is not None:
                todo.is_completed = todo_update.is_completed
            if todo_update.updated_by is not None:
                todo.updated_by = todo_update.updated_by
            todo.updated_at = datetime.now(timezone.utc)

            self.db_session.commit()
            self.db_session.refresh(todo)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.success,
                result=[ToDoResponseSchema(
                    id=todo.id,
                    title=todo.title,
                    description=todo.description,
                    is_completed=todo.is_completed,
                    created_at=todo.created_at.isoformat() if todo.created_at else None,
                    updated_at=todo.updated_at.isoformat() if todo.updated_at else None,
                    created_by=todo.created_by,
                    updated_by=todo.updated_by,
                )],
                status_code=status.HTTP_200_OK
            )
        except SQLAlchemyError as e:
            self.db_session.rollback()
            print(f"{GlobalConstants.api_response_messages.error_in} update_todo service: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.db_error,
                result=None,
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR
            )

    def delete_todo(self, todo_id: str):
        try:
            result = self.db_session.execute(
                select(ToDo).where(ToDo.id == todo_id)
            )
            todo = result.scalars().first()
            if not todo:
                return GlobalConstants.return_api_response(
                    message=f"ToDo with ID {todo_id} not found.",
                    result=[],
                    status_code=status.HTTP_400_BAD_REQUEST
                )
            self.db_session.delete(todo)
            self.db_session.commit()
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.success,
                result=None,
                status_code=status.HTTP_200_OK
            )
        except SQLAlchemyError as e:
            self.db_session.rollback()
            print(f"{GlobalConstants.api_response_messages.error_in} delete_todo service: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.db_error,
                result=None,
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR
            )



    def get_todos_by_created_by_id(self, created_by_id: str):
        try:
            user_exists = self.db_session.execute(
                select(User).where(User.id == created_by_id)
            ).scalars().first()
            if not user_exists:
                return GlobalConstants.return_api_response(
                    message=f"User with ID {created_by_id} not found.",
                    result=[],
                    status_code=status.HTTP_400_BAD_REQUEST
                )

            result = self.db_session.execute(
                select(ToDo).where(ToDo.created_by == created_by_id)
            )
            todos = result.scalars().all()
            
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.success,
                result=[ToDoResponseSchema(
                    id=todo.id,
                    title=todo.title,
                    description=todo.description,
                    is_completed=todo.is_completed,
                    created_at=todo.created_at.isoformat() if todo.created_at else None,
                    updated_at=todo.updated_at.isoformat() if todo.updated_at else None,
                    created_by=todo.created_by,
                    updated_by=todo.updated_by,
                ) for todo in todos],
                status_code=status.HTTP_200_OK
            )
        except SQLAlchemyError as e:
            print(f"{GlobalConstants.api_response_messages.error_in} get_todos_by_created_by_id service: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.db_error,
                result=None,
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR
            )