from sqlalchemy.orm import Session
from sqlalchemy.exc import SQLAlchemyError
from app.models.user_model import (
    User,
    UserCreateSchema,
    UserUpdateSchema,
    UserResponseSchema,
)
from fastapi import status
from datetime import datetime, timezone
from app.global_constants import GlobalConstants
from sqlalchemy.future import select
from uuid import UUID
import hashlib
from app.Oauth import hash_password  

class UserService:
    def __init__(self, db_session: Session):
        self.db_session = db_session


    def create_user(self, user_data: UserCreateSchema):
        try:

            # Check if email already exists
            existing_user = self.db_session.execute(
                select(User).where(User.email == user_data.email)
            ).scalars().first()
            if existing_user:
                return GlobalConstants.return_api_response(
                    message="Email already exists.",
                    result=[],
                    status_code=status.HTTP_400_BAD_REQUEST
                )
                
            # Use bcrypt to hash the password
            hashed_password = hash_password(user_data.password)

            new_user = User(
                username=user_data.username,
                email=user_data.email,
                full_name=user_data.full_name,
                hashed_password=hashed_password,
                created_at=datetime.now(timezone.utc),
                updated_at=datetime.now(timezone.utc),
                created_by=user_data.created_by,
                updated_by=user_data.created_by,
            )
            self.db_session.add(new_user)
            self.db_session.commit()
            self.db_session.refresh(new_user)

            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.accepted,
                result=[UserResponseSchema(
                    id=new_user.id,
                    username=new_user.username,
                    email=new_user.email,
                    full_name=new_user.full_name,
                    created_at=new_user.created_at.isoformat() if new_user.created_at else None,
                    updated_at=new_user.updated_at.isoformat() if new_user.updated_at else None,
                    created_by=new_user.created_by,
                    updated_by=new_user.updated_by,
                )],
                status_code=status.HTTP_201_CREATED
            )
        except SQLAlchemyError as e:
            self.db_session.rollback()
            print(f"{GlobalConstants.api_response_messages.error_in} create_user service: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.db_error,
                result=None,
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR
            )

    def get_all_users(self):
        try:
            result = self.db_session.execute(select(User))
            users = result.scalars().all()

            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.success,
                result=[UserResponseSchema(
                    id=user.id,
                    username=user.username,
                    email=user.email,
                    full_name=user.full_name,
                    created_at=user.created_at.isoformat() if user.created_at else None,
                    updated_at=user.updated_at.isoformat() if user.updated_at else None,
                    created_by=user.created_by,
                    updated_by=user.updated_by,
                ) for user in users],
                status_code=status.HTTP_200_OK
            )
        except SQLAlchemyError as e:
            print(f"{GlobalConstants.api_response_messages.error_in} get_all_users service: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.db_error,
                result=None,
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR
            )

    def get_user_by_id(self, user_id: UUID):
        try:
            result = self.db_session.execute(select(User).where(User.id == user_id))
            user = result.scalars().first()
            if not user:
                return GlobalConstants.return_api_response(
                    message=f"User with ID {user_id} not found.",
                    result=[],
                    status_code=status.HTTP_400_BAD_REQUEST
                )

            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.success,
                result=[UserResponseSchema(
                    id=user.id,
                    username=user.username,
                    email=user.email,
                    full_name=user.full_name,
                    created_at=user.created_at.isoformat() if user.created_at else None,
                    updated_at=user.updated_at.isoformat() if user.updated_at else None,
                    created_by=user.created_by,
                    updated_by=user.updated_by,
                )],
                status_code=status.HTTP_200_OK
            )
        except SQLAlchemyError as e:
            print(f"{GlobalConstants.api_response_messages.error_in} get_user_by_id service: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.db_error,
                result=None,
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR
            )

    def update_user(self, user_update: UserUpdateSchema):
        try:
            result = self.db_session.execute(select(User).where(User.id == user_update.user_id))
            user = result.scalars().first()
            if not user:
                return GlobalConstants.return_api_response(
                    message=f"User with ID {user_update.user_id} not found.",
                    result=[],
                    status_code=status.HTTP_400_BAD_REQUEST
                )
            if user_update.email is not None:
                user.email = user_update.email
            if user_update.full_name is not None:
                user.full_name = user_update.full_name
            if user_update.password is not None:
                user.hashed_password = hash_password(user_update.password)
            if user_update.updated_by is not None:
                user.updated_by = user_update.updated_by

            user.updated_at = datetime.now(timezone.utc)
            self.db_session.commit()
            self.db_session.refresh(user)

            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.success,
                result=[UserResponseSchema(
                    id=user.id,
                    username=user.username,
                    email=user.email,
                    full_name=user.full_name,
                    created_at=user.created_at.isoformat() if user.created_at else None,
                    updated_at=user.updated_at.isoformat() if user.updated_at else None,
                    created_by=user.created_by,
                    updated_by=user.updated_by,
                )],
                status_code=status.HTTP_200_OK
            )
        except SQLAlchemyError as e:
            self.db_session.rollback()
            print(f"{GlobalConstants.api_response_messages.error_in} update_user service: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.db_error,
                result=None,
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR
            )

    def delete_user(self, user_id: UUID):
        try:
            result = self.db_session.execute(select(User).where(User.id == user_id))
            user = result.scalars().first()
            if not user:
                return GlobalConstants.return_api_response(
                    message=f"User with ID {user_id} not found.",
                    result=[],
                    status_code=status.HTTP_400_BAD_REQUEST
                )
            self.db_session.delete(user)
            self.db_session.commit()
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.success,
                result=None,
                status_code=status.HTTP_200_OK
            )
        except SQLAlchemyError as e:
            self.db_session.rollback()
            print(f"{GlobalConstants.api_response_messages.error_in} delete_user service: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.db_error,
                result=None,
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR
            )
