import datetime
import uuid
import json
import os
from typing import Dict, List, Optional, Set, Tuple, Any, Union


class Person:
    """Base class for all people in the system."""
    
    def __init__(self, first_name: str, last_name: str, date_of_birth: str, contact_email: str):
        self.id = str(uuid.uuid4())
        self.first_name = first_name
        self.last_name = last_name
        self.full_name = f"{first_name} {last_name}"
        
        # Validate and set date of birth
        try:
            self.date_of_birth = datetime.datetime.strptime(date_of_birth, "%Y-%m-%d").date()
        except ValueError:
            raise ValueError("Date of birth must be in the format YYYY-MM-DD")
        
        # Validate email format (basic check)
        if "@" not in contact_email or "." not in contact_email:
            raise ValueError("Invalid email format")
        self.contact_email = contact_email
    
    @property
    def age(self) -> int:
        """Calculate age based on date of birth."""
        today = datetime.date.today()
        age = today.year - self.date_of_birth.year
        if (today.month, today.day) < (self.date_of_birth.month, self.date_of_birth.day):
            age -= 1
        return age
    
    def to_dict(self) -> Dict[str, Any]:
        """Convert person to dictionary for serialization."""
        return {
            "id": self.id,
            "first_name": self.first_name,
            "last_name": self.last_name,
            "date_of_birth": self.date_of_birth.strftime("%Y-%m-%d"),
            "contact_email": self.contact_email
        }
    
    def __str__(self) -> str:
        return f"{self.full_name} (ID: {self.id})"


class Course:
    """Represents an academic course."""
    
    def __init__(self, code: str, name: str, credits: int, description: str = ""):
        self.code = code.upper()  # Course code, e.g., "CS101"
        self.name = name
        
        # Validate credits
        if not isinstance(credits, int) or credits <= 0:
            raise ValueError("Credits must be a positive integer")
        self.credits = credits
        
        self.description = description
        self.students: Set[str] = set()  # Set of student IDs enrolled in the course
    
    def enroll_student(self, student_id: str) -> None:
        """Enroll a student in this course."""
        self.students.add(student_id)
    
    def drop_student(self, student_id: str) -> bool:
        """Drop a student from this course."""
        if student_id in self.students:
            self.students.remove(student_id)
            return True
        return False
    
    def get_enrollment_count(self) -> int:
        """Get the number of students enrolled in this course."""
        return len(self.students)
    
    def to_dict(self) -> Dict[str, Any]:
        """Convert course to dictionary for serialization."""
        return {
            "code": self.code,
            "name": self.name,
            "credits": self.credits,
            "description": self.description,
            "students": list(self.students)
        }
    
    def __str__(self) -> str:
        return f"{self.code}: {self.name} ({self.credits} credits)"


class Grade:
    """Represents a grade for a student in a specific course."""
    
    VALID_GRADES = {"A+", "A", "A-", "B+", "B", "B-", "C+", "C", "C-", "D+", "D", "D-", "F", "I", "W"}
    GRADE_POINTS = {
        "A+": 4.0, "A": 4.0, "A-": 3.7,
        "B+": 3.3, "B": 3.0, "B-": 2.7,
        "C+": 2.3, "C": 2.0, "C-": 1.7,
        "D+": 1.3, "D": 1.0, "D-": 0.7,
        "F": 0.0, "I": None, "W": None  # I = Incomplete, W = Withdrawn
    }
    
    def __init__(self, student_id: str, course_code: str, grade: str, semester: str, year: int):
        self.student_id = student_id
        self.course_code = course_code.upper()
        
        # Validate grade
        if grade not in self.VALID_GRADES:
            raise ValueError(f"Invalid grade. Must be one of {self.VALID_GRADES}")
        self.grade = grade
        
        # Validate semester (Fall, Spring, Summer)
        valid_semesters = {"Fall", "Spring", "Summer"}
        if semester not in valid_semesters:
            raise ValueError(f"Invalid semester. Must be one of {valid_semesters}")
        self.semester = semester
        
        # Validate year
        current_year = datetime.date.today().year
        if not isinstance(year, int) or year < 1900 or year > current_year + 1:
            raise ValueError(f"Invalid year. Must be between 1900 and {current_year + 1}")
        self.year = year
        
        # Record when the grade was added
        self.date_recorded = datetime.date.today()
    
    @property
    def grade_points(self) -> Optional[float]:
        """Get grade points associated with this grade."""
        return self.GRADE_POINTS[self.grade]
    
    @property
    def term(self) -> str:
        """Get formatted term string."""
        return f"{self.semester} {self.year}"
    
    def to_dict(self) -> Dict[str, Any]:
        """Convert grade to dictionary for serialization."""
        return {
            "student_id": self.student_id,
            "course_code": self.course_code,
            "grade": self.grade,
            "semester": self.semester,
            "year": self.year,
            "date_recorded": self.date_recorded.strftime("%Y-%m-%d")
        }
    
    def __str__(self) -> str:
        return f"{self.course_code}: {self.grade} ({self.term})"


class Student(Person):
    """Represents a student in the system."""
    
    def __init__(
        self, 
        first_name: str, 
        last_name: str, 
        date_of_birth: str, 
        contact_email: str,
        student_id: Optional[str] = None,
        major: str = "Undeclared",
        enrollment_date: Optional[str] = None
    ):
        super().__init__(first_name, last_name, date_of_birth, contact_email)
        
        # Use provided ID or generate a new one
        self.student_id = student_id if student_id else f"S{str(uuid.uuid4())[:8].upper()}"
        
        self.major = major
        
        # Set enrollment date to today if not provided
        if enrollment_date:
            try:
                self.enrollment_date = datetime.datetime.strptime(enrollment_date, "%Y-%m-%d").date()
            except ValueError:
                raise ValueError("Enrollment date must be in the format YYYY-MM-DD")
        else:
            self.enrollment_date = datetime.date.today()
        
        self.enrolled_courses: Set[str] = set()  # Set of course codes
        self.grades: List[Grade] = []
        self.status = "Active"  # Active, On Leave, Graduated, Withdrawn, etc.
    
    def enroll_in_course(self, course_code: str) -> None:
        """Enroll student in a course."""
        self.enrolled_courses.add(course_code.upper())
    
    def drop_course(self, course_code: str) -> bool:
        """Drop a course."""
        course_code = course_code.upper()
        if course_code in self.enrolled_courses:
            self.enrolled_courses.remove(course_code)
            return True
        return False
    
    def add_grade(self, grade: Grade) -> None:
        """Add a grade for a course."""
        if grade.student_id != self.student_id:
            raise ValueError("Grade does not belong to this student")
        self.grades.append(grade)
    
    def get_gpa(self) -> Optional[float]:
        """Calculate student's GPA."""
        valid_grades = [g for g in self.grades if g.grade_points is not None]
        if not valid_grades:
            return None
        
        total_points = sum(g.grade_points * self.get_course_credits(g.course_code) for g in valid_grades)
        total_credits = sum(self.get_course_credits(g.course_code) for g in valid_grades)
        
        if total_credits == 0:
            return None
        
        return round(total_points / total_credits, 2)
    
    def get_course_credits(self, course_code: str) -> int:
        """Helper method to get course credits (would normally look up in a course catalog)."""
        # This would ideally query the SMS system's course catalog
        # For now, we'll return a default value of 3
        return 3
    
    def get_transcript(self) -> List[Dict[str, Any]]:
        """Generate student transcript."""
        transcript = []
        for grade in sorted(self.grades, key=lambda g: (g.year, g.semester, g.course_code)):
            transcript.append({
                "course_code": grade.course_code,
                "grade": grade.grade,
                "term": grade.term,
                "credits": self.get_course_credits(grade.course_code),
                "grade_points": grade.grade_points
            })
        return transcript
    
    def to_dict(self) -> Dict[str, Any]:
        """Convert student to dictionary for serialization."""
        base_dict = super().to_dict()
        student_dict = {
            "student_id": self.student_id,
            "major": self.major,
            "enrollment_date": self.enrollment_date.strftime("%Y-%m-%d"),
            "enrolled_courses": list(self.enrolled_courses),
            "grades": [g.to_dict() for g in self.grades],
            "status": self.status
        }
        return {**base_dict, **student_dict}
    
    def __str__(self) -> str:
        return f"{self.full_name} (Student ID: {self.student_id}, Major: {self.major})"


class Instructor(Person):
    """Represents an instructor in the system."""
    
    def __init__(
        self, 
        first_name: str, 
        last_name: str, 
        date_of_birth: str, 
        contact_email: str,
        employee_id: Optional[str] = None,
        department: str = "",
        position: str = "Instructor",
        hire_date: Optional[str] = None
    ):
        super().__init__(first_name, last_name, date_of_birth, contact_email)
        
        # Use provided ID or generate a new one
        self.employee_id = employee_id if employee_id else f"E{str(uuid.uuid4())[:8].upper()}"
        
        self.department = department
        self.position = position
        
        # Set hire date to today if not provided
        if hire_date:
            try:
                self.hire_date = datetime.datetime.strptime(hire_date, "%Y-%m-%d").date()
            except ValueError:
                raise ValueError("Hire date must be in the format YYYY-MM-DD")
        else:
            self.hire_date = datetime.date.today()
        
        self.courses_taught: Set[str] = set()  # Set of course codes
    
    def assign_course(self, course_code: str) -> None:
        """Assign instructor to teach a course."""
        self.courses_taught.add(course_code.upper())
    
    def remove_course(self, course_code: str) -> bool:
        """Remove a course from instructor's teaching load."""
        course_code = course_code.upper()
        if course_code in self.courses_taught:
            self.courses_taught.remove(course_code)
            return True
        return False
    
    def to_dict(self) -> Dict[str, Any]:
        """Convert instructor to dictionary for serialization."""
        base_dict = super().to_dict()
        instructor_dict = {
            "employee_id": self.employee_id,
            "department": self.department,
            "position": self.position,
            "hire_date": self.hire_date.strftime("%Y-%m-%d"),
            "courses_taught": list(self.courses_taught)
        }
        return {**base_dict, **instructor_dict}
    
    def __str__(self) -> str:
        return f"{self.full_name} ({self.position}, {self.department})"


class StudentManagementSystem:
    """Main class for managing students, courses, and grades."""
    
    def __init__(self, data_file: str = "sms_data.json"):
        self.students: Dict[str, Student] = {}  # Map student_id to Student object
        self.courses: Dict[str, Course] = {}  # Map course_code to Course object
        self.instructors: Dict[str, Instructor] = {}  # Map employee_id to Instructor object
        self.data_file = data_file
        
        # Load data if file exists
        if os.path.exists(data_file):
            self.load_data()
    
    def add_student(self, student: Student) -> None:
        """Add a student to the system."""
        if student.student_id in self.students:
            raise ValueError(f"Student with ID {student.student_id} already exists")
        self.students[student.student_id] = student
    
    def get_student(self, student_id: str) -> Optional[Student]:
        """Get a student by ID."""
        return self.students.get(student_id)
    
    def add_course(self, course: Course) -> None:
        """Add a course to the system."""
        if course.code in self.courses:
            raise ValueError(f"Course with code {course.code} already exists")
        self.courses[course.code] = course
    
    def get_course(self, course_code: str) -> Optional[Course]:
        """Get a course by code."""
        return self.courses.get(course_code.upper())
    
    def add_instructor(self, instructor: Instructor) -> None:
        """Add an instructor to the system."""
        if instructor.employee_id in self.instructors:
            raise ValueError(f"Instructor with ID {instructor.employee_id} already exists")
        self.instructors[instructor.employee_id] = instructor
    
    def get_instructor(self, employee_id: str) -> Optional[Instructor]:
        """Get an instructor by ID."""
        return self.instructors.get(employee_id)
    
    def enroll_student_in_course(self, student_id: str, course_code: str) -> bool:
        """Enroll a student in a course."""
        student = self.get_student(student_id)
        course = self.get_course(course_code.upper())
        
        if not student:
            raise ValueError(f"Student with ID {student_id} not found")
        if not course:
            raise ValueError(f"Course with code {course_code} not found")
        
        student.enroll_in_course(course.code)
        course.enroll_student(student_id)
        return True
    
    def drop_student_from_course(self, student_id: str, course_code: str) -> bool:
        """Drop a student from a course."""
        student = self.get_student(student_id)
        course = self.get_course(course_code.upper())
        
        if not student or not course:
            return False
        
        return student.drop_course(course.code) and course.drop_student(student_id)
    
    def assign_grade(
        self, 
        student_id: str, 
        course_code: str, 
        grade: str, 
        semester: str, 
        year: int
    ) -> bool:
        """Assign a grade to a student for a course."""
        student = self.get_student(student_id)
        course = self.get_course(course_code.upper())
        
        if not student:
            raise ValueError(f"Student with ID {student_id} not found")
        if not course:
            raise ValueError(f"Course with code {course_code} not found")
        
        # Create and add grade
        grade_obj = Grade(student_id, course_code, grade, semester, year)
        student.add_grade(grade_obj)
        return True
    
    def get_student_gpa(self, student_id: str) -> Optional[float]:
        """Get a student's GPA."""
        student = self.get_student(student_id)
        if not student:
            raise ValueError(f"Student with ID {student_id} not found")
        return student.get_gpa()
    
    def get_student_transcript(self, student_id: str) -> List[Dict[str, Any]]:
        """Get a student's transcript."""
        student = self.get_student(student_id)
        if not student:
            raise ValueError(f"Student with ID {student_id} not found")
        return student.get_transcript()
    
    def assign_instructor_to_course(self, employee_id: str, course_code: str) -> bool:
        """Assign an instructor to teach a course."""
        instructor = self.get_instructor(employee_id)
        course = self.get_course(course_code.upper())
        
        if not instructor:
            raise ValueError(f"Instructor with ID {employee_id} not found")
        if not course:
            raise ValueError(f"Course with code {course_code} not found")
        
        instructor.assign_course(course.code)
        return True
    
    def generate_course_roster(self, course_code: str) -> List[Dict[str, Any]]:
        """Generate a roster of students for a course."""
        course = self.get_course(course_code.upper())
        
        if not course:
            raise ValueError(f"Course with code {course_code} not found")
        
        roster = []
        for student_id in course.students:
            student = self.get_student(student_id)
            if student:
                roster.append({
                    "student_id": student.student_id,
                    "name": student.full_name,
                    "email": student.contact_email,
                    "major": student.major
                })
        
        return roster
    
    def find_students_by_name(self, name: str) -> List[Student]:
        """Find students by name (case-insensitive partial match)."""
        name = name.lower()
        return [
            student for student in self.students.values()
            if name in student.full_name.lower()
        ]
    
    def find_courses_by_name(self, name: str) -> List[Course]:
        """Find courses by name (case-insensitive partial match)."""
        name = name.lower()
        return [
            course for course in self.courses.values()
            if name in course.name.lower()
        ]
    
    def get_students_by_major(self, major: str) -> List[Student]:
        """Get all students in a specific major."""
        return [
            student for student in self.students.values()
            if student.major.lower() == major.lower()
        ]
    
    def get_instructors_by_department(self, department: str) -> List[Instructor]:
        """Get all instructors in a specific department."""
        return [
            instructor for instructor in self.instructors.values()
            if instructor.department.lower() == department.lower()
        ]
    
    def save_data(self) -> None:
        """Save all data to file."""
        data = {
            "students": {sid: student.to_dict() for sid, student in self.students.items()},
            "courses": {code: course.to_dict() for code, course in self.courses.items()},
            "instructors": {eid: instructor.to_dict() for eid, instructor in self.instructors.items()}
        }
        
        with open(self.data_file, 'w') as f:
            json.dump(data, f, indent=2)
    
    def load_data(self) -> None:
        """Load data from file."""
        try:
            with open(self.data_file, 'r') as f:
                data = json.load(f)
            
            # Load courses first
            for code, course_data in data.get("courses", {}).items():
                course = Course(
                    code=course_data["code"],
                    name=course_data["name"],
                    credits=course_data["credits"],
                    description=course_data.get("description", "")
                )
                # Add student IDs to the course
                for student_id in course_data.get("students", []):
                    course.enroll_student(student_id)
                self.courses[code] = course
            
            # Load students
            for sid, student_data in data.get("students", {}).items():
                # Create student object
                student = Student(
                    first_name=student_data["first_name"],
                    last_name=student_data["last_name"],
                    date_of_birth=student_data["date_of_birth"],
                    contact_email=student_data["contact_email"],
                    student_id=student_data["student_id"],
                    major=student_data.get("major", "Undeclared"),
                    enrollment_date=student_data.get("enrollment_date")
                )
                
                # Add enrolled courses
                for course_code in student_data.get("enrolled_courses", []):
                    student.enroll_in_course(course_code)
                
                # Add grades
                for grade_data in student_data.get("grades", []):
                    grade = Grade(
                        student_id=grade_data["student_id"],
                        course_code=grade_data["course_code"],
                        grade=grade_data["grade"],
                        semester=grade_data["semester"],
                        year=grade_data["year"]
                    )
                    student.add_grade(grade)
                
                student.status = student_data.get("status", "Active")
                self.students[sid] = student
            
            # Load instructors
            for eid, instructor_data in data.get("instructors", {}).items():
                instructor = Instructor(
                    first_name=instructor_data["first_name"],
                    last_name=instructor_data["last_name"],
                    date_of_birth=instructor_data["date_of_birth"],
                    contact_email=instructor_data["contact_email"],
                    employee_id=instructor_data["employee_id"],
                    department=instructor_data.get("department", ""),
                    position=instructor_data.get("position", "Instructor"),
                    hire_date=instructor_data.get("hire_date")
                )
                
                # Add courses taught
                for course_code in instructor_data.get("courses_taught", []):
                    instructor.assign_course(course_code)
                
                self.instructors[eid] = instructor
                
        except Exception as e:
            print(f"Error loading data: {e}")
            # Initialize empty data structures
            self.students = {}
            self.courses = {}
            self.instructors = {}


# Example usage
if __name__ == "__main__":
    # Create SMS instance
    sms = StudentManagementSystem()
    
    # Add some courses
    cs101 = Course("CS101", "Introduction to Computer Science", 3, "Basic programming concepts")
    cs102 = Course("CS102", "Data Structures", 4, "Fundamental data structures and algorithms")
    math201 = Course("MATH201", "Calculus I", 4, "Limits, derivatives, and integrals")
    
    sms.add_course(cs101)
    sms.add_course(cs102)
    sms.add_course(math201)
    
    # Add some students
    alice = Student(
        first_name="Alice",
        last_name="Johnson",
        date_of_birth="2000-05-15",
        contact_email="alice@example.com",
        major="Computer Science"
    )
    
    bob = Student(
        first_name="Bob",
        last_name="Smith",
        date_of_birth="2001-02-20",
        contact_email="bob@example.com",
        major="Mathematics"
    )
    
    sms.add_student(alice)
    sms.add_student(bob)
    
    # Add an instructor
    prof_brown = Instructor(
        first_name="David",
        last_name="Brown",
        date_of_birth="1975-08-10",
        contact_email="brown@example.edu",
        department="Computer Science",
        position="Professor"
    )
    
    sms.add_instructor(prof_brown)
    
    # Assign instructor to courses
    sms.assign_instructor_to_course(prof_brown.employee_id, "CS101")
    sms.assign_instructor_to_course(prof_brown.employee_id, "CS102")
    
    # Enroll students in courses
    sms.enroll_student_in_course(alice.student_id, "CS101")
    sms.enroll_student_in_course(alice.student_id, "MATH201")
    sms.enroll_student_in_course(bob.student_id, "CS101")
    sms.enroll_student_in_course(bob.student_id, "CS102")
    
    # Assign grades
    sms.assign_grade(alice.student_id, "CS101", "A", "Fall", 2023)
    sms.assign_grade(bob.student_id, "CS101", "B+", "Fall", 2023)
    
    # Generate a course roster
    roster = sms.generate_course_roster("CS101")
    print("\nCS101 Course Roster:")
    for student in roster:
        print(f"- {student['name']} ({student['student_id']}), Major: {student['major']}")
    
    # Get student GPA
    alice_gpa = sms.get_student_gpa(alice.student_id)
    print(f"\nAlice's GPA: {alice_gpa}")
    
    # Get student transcript
    alice_transcript = sms.get_student_transcript(alice.student_id)
    print("\nAlice's Transcript:")
    for entry in alice_transcript:
        print(f"- {entry['course_code']}: {entry['grade']} ({entry['term']})")
    
    # Save data
    sms.save_data()
    print("\nData saved to sms_data.json")


class SMSInterface:
    """Command-line interface for the Student Management System."""
    
    def __init__(self, sms: StudentManagementSystem):
        self.sms = sms
        self.running = True
    
    def display_menu(self) -> None:
        """Display the main menu."""
        print("\n===== Student Management System =====")
        print("1. Student Management")
        print("2. Course Management")
        print("3. Instructor Management")
        print("4. Enrollment Management")
        print("5. Grade Management")
        print("6. Reports")
        print("7. Save Data")
        print("8. Exit")
        print("=====================================")
    
    def run(self) -> None:
        """Run the interface."""
        while self.running:
            self.display_menu()
            choice = input("Enter your choice (1-8): ")
            
            if choice == "1":
                self.student_menu()
            elif choice == "2":
                self.course_menu()
            elif choice == "3":
                self.instructor_menu()
            elif choice == "4":
                self.enrollment_menu()
            elif choice == "5":
                self.grade_menu()
            elif choice == "6":
                self.reports_menu()
            elif choice == "7":
                self.sms.save_data()
                print("Data saved successfully!")
            elif choice == "8":
                self.running = False
                print("Exiting Student Management System. Goodbye!")
            else:
                print("Invalid choice. Please try again.")
    
    def student_menu(self) -> None:
        """Display student management menu."""
        while True:
            print("\n----- Student Management -----")
            print("1. Add Student")
            print("2. View Student Details")
            print("3. Update Student Information")
            print("4. Search Students by Name")
            print("5. List Students by Major")
            print("6. Back to Main Menu")
            print("-----------------------------")
            
            choice = input("Enter your choice (1-6): ")
            
            if choice == "1":
                self.add_student()
            elif choice == "2":
                self.view_student()
            elif choice == "3":
                self.update_student()
            elif choice == "4":
                self.search_students_by_name()
            elif choice == "5":
                self.list_students_by_major()
            elif choice == "6":
                break
            else:
                print("Invalid choice. Please try again.")
    
    def add_student(self) -> None:
        """Add a new student."""
        print("\nEnter student information:")
        first_name = input("First Name: ")
        last_name = input("Last Name: ")
        
        # Validate date of birth
        while True:
            date_of_birth = input("Date of Birth (YYYY-MM-DD): ")
            try:
                datetime.datetime.strptime(date_of_birth, "%Y-%m-%d")
                break
            except ValueError:
                print("Invalid date format. Please use YYYY-MM-DD.")
        
        contact_email = input("Email: ")
        major = input("Major (or leave blank for Undeclared): ") or "Undeclared"
        
        try:
            student = Student(
                first_name=first_name,
                last_name=last_name,
                date_of_birth=date_of_birth,
                contact_email=contact_email,
                major=major
            )
            self.sms.add_student(student)
            print(f"Student added successfully! Student ID: {student.student_id}")
        except ValueError as e:
            print(f"Error: {e}")
    
    def view_student(self) -> None:
        """View student details."""
        student_id = input("Enter Student ID: ")
        student = self.sms.get_student(student_id)
        
        if student:
            print("\nStudent Information:")
            print(f"Name: {student.full_name}")
            print(f"Student ID: {student.student_id}")
            print(f"Major: {student.major}")
            print(f"Date of Birth: {student.date_of_birth}")
            print(f"Age: {student.age}")
            print(f"Email: {student.contact_email}")
            print(f"Enrollment Date: {student.enrollment_date}")
            print(f"Status: {student.status}")
            
            # Show enrolled courses
            print("\nEnrolled Courses:")
            if not student.enrolled_courses:
                print("None")
            else:
                for course_code in student.enrolled_courses:
                    course = self.sms.get_course(course_code)
                    if course:
                        print(f"- {course.code}: {course.name}")
            
            # Show GPA
            gpa = student.get_gpa()
            print(f"\nCurrent GPA: {gpa if gpa is not None else 'N/A'}")
        else:
            print(f"Student with ID {student_id} not found.")
    
    def update_student(self) -> None:
        """Update student information."""
        student_id = input("Enter Student ID: ")
        student = self.sms.get_student(student_id)
        
        if student:
            print("\nUpdate Student Information (leave blank to keep current value):")
            
            # Get new values with current values as defaults
            first_name = input(f"First Name [{student.first_name}]: ") or student.first_name
            last_name = input(f"Last Name [{student.last_name}]: ") or student.last_name
            email = input(f"Email [{student.contact_email}]: ") or student.contact_email
            major = input(f"Major [{student.major}]: ") or student.major
            status = input(f"Status [{student.status}]: ") or student.status
            
            # Update student information
            student.first_name = first_name
            student.last_name = last_name
            student.full_name = f"{first_name} {last_name}"
            student.contact_email = email
            student.major = major
            student.status = status
            
            print("Student information updated successfully!")
        else:
            print(f"Student with ID {student_id} not found.")
    
    def search_students_by_name(self) -> None:
        """Search for students by name."""
        name = input("Enter name to search: ")
        students = self.sms.find_students_by_name(name)
        
        if students:
            print(f"\nFound {len(students)} student(s):")
            for student in students:
                print(f"- {student.full_name} (ID: {student.student_id}, Major: {student.major})")
        else:
            print(f"No students found matching '{name}'.")
    
    def list_students_by_major(self) -> None:
        """List students by major."""
        major = input("Enter major: ")
        students = self.sms.get_students_by_major(major)
        
        if students:
            print(f"\nStudents majoring in {major}:")
            for student in students:
                print(f"- {student.full_name} (ID: {student.student_id})")
        else:
            print(f"No students found majoring in {major}.")
    
    def course_menu(self) -> None:
        """Display course management menu."""
        while True:
            print("\n----- Course Management -----")
            print("1. Add Course")
            print("2. View Course Details")
            print("3. Search Courses")
            print("4. List All Courses")
            print("5. Back to Main Menu")
            print("---------------------------")
            
            choice = input("Enter your choice (1-5): ")
            
            if choice == "1":
                self.add_course()
            elif choice == "2":
                self.view_course()
            elif choice == "3":
                self.search_courses()
            elif choice == "4":
                self.list_all_courses()
            elif choice == "5":
                break
            else:
                print("Invalid choice. Please try again.")
    
    def add_course(self) -> None:
        """Add a new course."""
        print("\nEnter course information:")
        code = input("Course Code (e.g., CS101): ")
        name = input("Course Name: ")
        
        # Validate credits
        while True:
            try:
                credits = int(input("Credits: "))
                if credits <= 0:
                    print("Credits must be a positive integer.")
                    continue
                break
            except ValueError:
                print("Please enter a valid number.")
        
        description = input("Description (optional): ")
        
        try:
            course = Course(code, name, credits, description)
            self.sms.add_course(course)
            print(f"Course {course.code} added successfully!")
        except ValueError as e:
            print(f"Error: {e}")
    
    def view_course(self) -> None:
        """View course details."""
        code = input("Enter Course Code: ")
        course = self.sms.get_course(code)
        
        if course:
            print("\nCourse Information:")
            print(f"Code: {course.code}")
            print(f"Name: {course.name}")
            print(f"Credits: {course.credits}")
            print(f"Description: {course.description}")
            print(f"Enrollment Count: {course.get_enrollment_count()}")
            
            # Show enrolled students
            if course.students:
                print("\nEnrolled Students:")
                for student_id in course.students:
                    student = self.sms.get_student(student_id)
                    if student:
                        print(f"- {student.full_name} (ID: {student.student_id})")
            else:
                print("\nNo students enrolled in this course.")
        else:
            print(f"Course with code {code} not found.")
    
    def search_courses(self) -> None:
        """Search for courses by name."""
        name = input("Enter course name to search: ")
        courses = self.sms.find_courses_by_name(name)
        
        if courses:
            print(f"\nFound {len(courses)} course(s):")
            for course in courses:
                print(f"- {course.code}: {course.name} ({course.credits} credits)")
        else:
            print(f"No courses found matching '{name}'.")
    
    def list_all_courses(self) -> None:
        """List all courses."""
        if self.sms.courses:
            print("\nAll Courses:")
            for course in self.sms.courses.values():
                print(f"- {course.code}: {course.name} ({course.credits} credits)")
        else:
            print("No courses available.")
    
    def instructor_menu(self) -> None:
        """Display instructor management menu."""
        while True:
            print("\n----- Instructor Management -----")
            print("1. Add Instructor")
            print("2. View Instructor Details")
            print("3. List Instructors by Department")
            print("4. Assign Course to Instructor")
            print("5. Back to Main Menu")
            print("-------------------------------")
            
            choice = input("Enter your choice (1-5): ")
            
            if choice == "1":
                self.add_instructor()
            elif choice == "2":
                self.view_instructor()
            elif choice == "3":
                self.list_instructors_by_department()
            elif choice == "4":
                self.assign_course_to_instructor()
            elif choice == "5":
                break
            else:
                print("Invalid choice. Please try again.")
    
    def add_instructor(self) -> None:
        """Add a new instructor."""
        print("\nEnter instructor information:")
        first_name = input("First Name: ")
        last_name = input("Last Name: ")
        
        # Validate date of birth
        while True:
            date_of_birth = input("Date of Birth (YYYY-MM-DD): ")
            try:
                datetime.datetime.strptime(date_of_birth, "%Y-%m-%d")
                break
            except ValueError:
                print("Invalid date format. Please use YYYY-MM-DD.")
        
        contact_email = input("Email: ")
        department = input("Department: ")
        position = input("Position (default: Instructor): ") or "Instructor"
        
        try:
            instructor = Instructor(
                first_name=first_name,
                last_name=last_name,
                date_of_birth=date_of_birth,
                contact_email=contact_email,
                department=department,
                position=position
            )
            self.sms.add_instructor(instructor)
            print(f"Instructor added successfully! Employee ID: {instructor.employee_id}")
        except ValueError as e:
            print(f"Error: {e}")
    
    def view_instructor(self) -> None:
        """View instructor details."""
        employee_id = input("Enter Employee ID: ")
        instructor = self.sms.get_instructor(employee_id)
        
        if instructor:
            print("\nInstructor Information:")
            print(f"Name: {instructor.full_name}")
            print(f"Employee ID: {instructor.employee_id}")
            print(f"Department: {instructor.department}")
            print(f"Position: {instructor.position}")
            print(f"Email: {instructor.contact_email}")
            print(f"Hire Date: {instructor.hire_date}")
            
            # Show courses taught
            print("\nCourses Taught:")
            if not instructor.courses_taught:
                print("None")
            else:
                for course_code in instructor.courses_taught:
                    course = self.sms.get_course(course_code)
                    if course:
                        print(f"- {course.code}: {course.name}")
        else:
            print(f"Instructor with ID {employee_id} not found.")
    
    def list_instructors_by_department(self) -> None:
        """List instructors by department."""
        department = input("Enter department: ")
        instructors = self.sms.get_instructors_by_department(department)
        
        if instructors:
            print(f"\nInstructors in {department} department:")
            for instructor in instructors:
                print(f"- {instructor.full_name} ({instructor.position}, ID: {instructor.employee_id})")
        else:
            print(f"No instructors found in {department} department.")
    
    def assign_course_to_instructor(self) -> None:
        """Assign a course to an instructor."""
        employee_id = input("Enter Instructor Employee ID: ")
        instructor = self.sms.get_instructor(employee_id)
        
        if not instructor:
            print(f"Instructor with ID {employee_id} not found.")
            return
        
        course_code = input("Enter Course Code: ")
        course = self.sms.get_course(course_code)
        
        if not course:
            print(f"Course with code {course_code} not found.")
            return
        
        if self.sms.assign_instructor_to_course(employee_id, course_code):
            print(f"Course {course.code} assigned to {instructor.full_name} successfully!")
        else:
            print("Failed to assign course to instructor.")
    
    def enrollment_menu(self) -> None:
        """Display enrollment management menu."""
        while True:
            print("\n----- Enrollment Management -----")
            print("1. Enroll Student in Course")
            print("2. Drop Student from Course")
            print("3. View Course Roster")
            print("4. Back to Main Menu")
            print("--------------------------------")
            
            choice = input("Enter your choice (1-4): ")
            
            if choice == "1":
                self.enroll_student_in_course()
            elif choice == "2":
                self.drop_student_from_course()
            elif choice == "3":
                self.view_course_roster()
            elif choice == "4":
                break
            else:
                print("Invalid choice. Please try again.")
    
    def enroll_student_in_course(self) -> None:
        """Enroll a student in a course."""
        student_id = input("Enter Student ID: ")
        student = self.sms.get_student(student_id)
        
        if not student:
            print(f"Student with ID {student_id} not found.")
            return
        
        course_code = input("Enter Course Code: ")
        course = self.sms.get_course(course_code)
        
        if not course:
            print(f"Course with code {course_code} not found.")
            return
        
        try:
            if self.sms.enroll_student_in_course(student_id, course_code):
                print(f"{student.full_name} enrolled in {course.code} successfully!")
        except ValueError as e:
            print(f"Error: {e}")
    
    def drop_student_from_course(self) -> None:
        """Drop a student from a course."""
        student_id = input("Enter Student ID: ")
        student = self.sms.get_student(student_id)
        
        if not student:
            print(f"Student with ID {student_id} not found.")
            return
        
        course_code = input("Enter Course Code: ")
        course = self.sms.get_course(course_code)
        
        if not course:
            print(f"Course with code {course_code} not found.")
            return
        
        if self.sms.drop_student_from_course(student_id, course_code):
            print(f"{student.full_name} dropped from {course.code} successfully!")
        else:
            print("Failed to drop student from course.")
    
    def view_course_roster(self) -> None:
        """View a course roster."""
        course_code = input("Enter Course Code: ")
        try:
            roster = self.sms.generate_course_roster(course_code)
            
            if roster:
                print(f"\nRoster for course {course_code}:")
                print(f"Total students: {len(roster)}")
                for i, student in enumerate(roster, 1):
                    print(f"{i}. {student['name']} (ID: {student['student_id']}, Major: {student['major']})")
            else:
                print(f"No students enrolled in course {course_code}.")
        except ValueError as e:
            print(f"Error: {e}")
    
    def grade_menu(self) -> None:
        """Display grade management menu."""
        while True:
            print("\n----- Grade Management -----")
            print("1. Assign Grade")
            print("2. View Student Grades")
            print("3. Calculate Student GPA")
            print("4. Back to Main Menu")
            print("---------------------------")
            
            choice = input("Enter your choice (1-4): ")
            
            if choice == "1":
                self.assign_grade()
            elif choice == "2":
                self.view_student_grades()
            elif choice == "3":
                self.calculate_student_gpa()
            elif choice == "4":
                break
            else:
                print("Invalid choice. Please try again.")
    
    def assign_grade(self) -> None:
        """Assign a grade to a student for a course."""
        student_id = input("Enter Student ID: ")
        student = self.sms.get_student(student_id)
        
        if not student:
            print(f"Student with ID {student_id} not found.")
            return
        
        course_code = input("Enter Course Code: ")
        course = self.sms.get_course(course_code)
        
        if not course:
            print(f"Course with code {course_code} not found.")
            return
        
        # Check if student is enrolled in the course
        if course_code.upper() not in student.enrolled_courses:
            enroll = input(f"{student.full_name} is not enrolled in {course_code}. Enroll now? (y/n): ")
            if enroll.lower() == 'y':
                self.sms.enroll_student_in_course(student_id, course_code)
            else:
                return
        
        # Show valid grades
        print("\nValid grades: A+, A, A-, B+, B, B-, C+, C, C-, D+, D, D-, F, I, W")
        grade = input("Enter Grade: ").upper()
        
        if grade not in Grade.VALID_GRADES:
            print(f"Invalid grade. Must be one of {Grade.VALID_GRADES}")
            return
        
        # Get semester and year
        print("\nSelect Semester:")
        print("1. Fall")
        print("2. Spring")
        print("3. Summer")
        
        semester_choice = input("Enter choice (1-3): ")
        if semester_choice == "1":
            semester = "Fall"
        elif semester_choice == "2":
            semester = "Spring"
        elif semester_choice == "3":
            semester = "Summer"
        else:
            print("Invalid choice.")
            return
        
        # Get year
        current_year = datetime.date.today().year
        while True:
            try:
                year = int(input(f"Enter Year ({current_year-5}-{current_year+1}): "))
                if year < current_year - 5 or year > current_year + 1:
                    print(f"Year must be between {current_year-5} and {current_year+1}.")
                    continue
                break
            except ValueError:
                print("Please enter a valid year.")
        
        try:
            if self.sms.assign_grade(student_id, course_code, grade, semester, year):
                print(f"Grade {grade} assigned to {student.full_name} for {course_code} successfully!")
        except ValueError as e:
            print(f"Error: {e}")
    
    def view_student_grades(self) -> None:
        """View a student's grades."""
        student_id = input("Enter Student ID: ")
        student = self.sms.get_student(student_id)
        
        if not student:
            print(f"Student with ID {student_id} not found.")
            return
        
        transcript = self.sms.get_student_transcript(student_id)
        
        if transcript:
            print(f"\nGrades for {student.full_name}:")
            for entry in transcript:
                print(f"- {entry['course_code']}: {entry['grade']} ({entry['term']}, {entry['credits']} credits)")
        else:
            print(f"No grades recorded for {student.full_name}.")
    
    def calculate_student_gpa(self) -> None:
        """Calculate a student's GPA."""
        student_id = input("Enter Student ID: ")
        
        try:
            gpa = self.sms.get_student_gpa(student_id)
            student = self.sms.get_student(student_id)
            
            if gpa is not None:
                print(f"\n{student.full_name}'s GPA: {gpa}")
            else:
                print(f"No GPA available for {student.full_name} (no graded courses).")
        except ValueError as e:
            print(f"Error: {e}")
    
    def reports_menu(self) -> None:
        """Display reports menu."""
        while True:
            print("\n----- Reports -----")
            print("1. Student Transcript")
            print("2. Course Enrollment Summary")
            print("3. Department Faculty List")
            print("4. GPA Summary by Major")
            print("5. Back to Main Menu")
            print("-------------------")
            
            choice = input("Enter your choice (1-5): ")
            
            if choice == "1":
                self.generate_student_transcript()
            elif choice == "2":
                self.generate_course_enrollment_summary()
            elif choice == "3":
                self.generate_department_faculty_list()
            elif choice == "4":
                self.generate_gpa_summary_by_major()
            elif choice == "5":
                break
            else:
                print("Invalid choice. Please try again.")
    
    def generate_student_transcript(self) -> None:
        """Generate a student transcript."""
        student_id = input("Enter Student ID: ")
        student = self.sms.get_student(student_id)
        
        if not student:
            print(f"Student with ID {student_id} not found.")
            return
        
        transcript = self.sms.get_student_transcript(student_id)
        gpa = self.sms.get_student_gpa(student_id)
        
        print("\n" + "=" * 40)
        print(f"OFFICIAL TRANSCRIPT - {student.full_name}")
        print("=" * 40)
        print(f"Student ID: {student.student_id}")
        print(f"Major: {student.major}")
        print(f"Enrollment Date: {student.enrollment_date}")
        print("-" * 40)
        
        if transcript:
            # Group by term
            terms = {}
            for entry in transcript:
                if entry['term'] not in terms:
                    terms[entry['term']] = []
                terms[entry['term']].append(entry)
            
            # Print each term
            for term in sorted(terms.keys()):
                print(f"\n{term}:")
                print("-" * 40)
                term_credits = 0
                term_points = 0
                
                for entry in terms[term]:
                    print(f"{entry['course_code']:<10} {entry['credits']} credits  Grade: {entry['grade']}")
                    if entry['grade_points'] is not None:
                        term_credits += entry['credits']
                        term_points += entry['credits'] * entry['grade_points']
                
                # Calculate term GPA
                if term_credits > 0:
                    term_gpa = round(term_points / term_credits, 2)
                    print(f"Term GPA: {term_gpa}")
            
            print("\n" + "-" * 40)
            print(f"Cumulative GPA: {gpa if gpa is not None else 'N/A'}")
        else:
            print("No courses completed.")
        
        print("=" * 40)
    
    def generate_course_enrollment_summary(self) -> None:
        """Generate a summary of course enrollments."""
        if not self.sms.courses:
            print("No courses available.")
            return
        
        print("\n" + "=" * 50)
        print("COURSE ENROLLMENT SUMMARY")
        print("=" * 50)
        print(f"{'Course Code':<12} {'Course Name':<30} {'Enrollment':<10} {'Credits':<8}")
        print("-" * 50)
        
        total_students = 0
        for course in sorted(self.sms.courses.values(), key=lambda c: c.code):
            enrollment = course.get_enrollment_count()
            total_students += enrollment
            print(f"{course.code:<12} {course.name[:30]:<30} {enrollment:<10} {course.credits:<8}")
        
        print("-" * 50)
        print(f"Total Courses: {len(self.sms.courses)}")
        print(f"Total Enrollments: {total_students}")
        print("=" * 50)
    
    def generate_department_faculty_list(self) -> None:
        """Generate a list of faculty by department."""
        if not self.sms.instructors:
            print("No instructors available.")
            return
        
        # Group instructors by department
        departments = {}
        for instructor in self.sms.instructors.values():
            dept = instructor.department
            if not dept:
                dept = "No Department"
            
            if dept not in departments:
                departments[dept] = []
            departments[dept].append(instructor)
        
        print("\n" + "=" * 60)
        print("FACULTY BY DEPARTMENT")
        print("=" * 60)
        
        for dept, instructors in sorted(departments.items()):
            print(f"\n{dept}:")
            print("-" * 30)
            for instructor in sorted(instructors, key=lambda i: i.last_name):
                courses = len(instructor.courses_taught)
                print(f"- {instructor.full_name:<25} {instructor.position:<15} Courses: {courses}")
        
        print("\n" + "=" * 60)
    
    def generate_gpa_summary_by_major(self) -> None:
        """Generate a GPA summary by major."""
        if not self.sms.students:
            print("No students available.")
            return
        
        # Group students by major
        majors = {}
        for student in self.sms.students.values():
            major = student.major
            if major not in majors:
                majors[major] = []
            majors[major].append(student)
        
        print("\n" + "=" * 60)
        print("GPA SUMMARY BY MAJOR")
        print("=" * 60)
        print(f"{'Major':<20} {'Students':<10} {'Avg GPA':<10} {'Min GPA':<10} {'Max GPA':<10}")
        print("-" * 60)
        
        for major, students in sorted(majors.items()):
            # Calculate GPA statistics
            gpas = [s.get_gpa() for s in students if s.get_gpa() is not None]
            
            if gpas:
                avg_gpa = round(sum(gpas) / len(gpas), 2)
                min_gpa = round(min(gpas), 2)
                max_gpa = round(max(gpas), 2)
                print(f"{major[:20]:<20} {len(students):<10} {avg_gpa:<10} {min_gpa:<10} {max_gpa:<10}")
            else:
                print(f"{major[:20]:<20} {len(students):<10} {'N/A':<10} {'N/A':<10} {'N/A':<10}")
        
        print("=" * 60)


# Demo function to run the SMS interface
def run_demo():
    """Run a demonstration of the Student Management System."""
    sms = StudentManagementSystem()
    
    # Add some sample data
    print("Initializing Student Management System with sample data...")
    
    # Add courses
    courses = [
        Course("CS101", "Introduction to Programming", 3, "Basic programming concepts using Python"),
        Course("CS102", "Data Structures", 4, "Fundamental data structures and algorithms"),
        Course("MATH101", "College Algebra", 3, "Basic algebraic operations"),
        Course("MATH201", "Calculus I", 4, "Limits, derivatives, and integrals"),
        Course("ENGL101", "Composition", 3, "Academic writing and research")
    ]
    
    for course in courses:
        try:
            sms.add_course(course)
        except ValueError:
            pass  # Skip duplicates
    
    # Add students
    students = [
        Student("John", "Smith", "2000-05-15", "john@example.com", major="Computer Science"),
        Student("Emma", "Johnson", "2001-03-22", "emma@example.com", major="Mathematics"),
        Student("Michael", "Davis", "1999-11-08", "michael@example.com", major="Computer Science"),
        Student("Sophia", "Brown", "2002-01-30", "sophia@example.com", major="English")
    ]
    
    for student in students:
        try:
            sms.add_student(student)
        except ValueError:
            pass  # Skip duplicates
    
    # Add instructors
    instructors = [
        Instructor("Robert", "Wilson", "1975-07-12", "rwilson@example.com", department="Computer Science", position="Professor"),
        Instructor("Jennifer", "Lee", "1980-09-18", "jlee@example.com", department="Mathematics", position="Associate Professor"),
        Instructor("William", "Anderson", "1968-04-25", "wanderson@example.com", department="English", position="Professor")
    ]
    
    for instructor in instructors:
        try:
            sms.add_instructor(instructor)
        except ValueError:
            pass  # Skip duplicates
    
    # Assign courses to instructors
    for instructor in instructors:
        if instructor.department == "Computer Science":
            sms.assign_instructor_to_course(instructor.employee_id, "CS101")
            sms.assign_instructor_to_course(instructor.employee_id, "CS102")
        elif instructor.department == "Mathematics":
            sms.assign_instructor_to_course(instructor.employee_id, "MATH101")
            sms.assign_instructor_to_course(instructor.employee_id, "MATH201")
        elif instructor.department == "English":
            sms.assign_instructor_to_course(instructor.employee_id, "ENGL101")
    
    # Enroll students in courses
    for student in students:
        if student.major == "Computer Science":
            sms.enroll_student_in_course(student.student_id, "CS101")
            sms.enroll_student_in_course(student.student_id, "CS102")
            sms.enroll_student_in_course(student.student_id, "MATH101")
            sms.enroll_student_in_course(student.student_id, "ENGL101")
        elif student.major == "Mathematics":
            sms.enroll_student_in_course(student.student_id, "MATH101")
            sms.enroll_student_in_course(student.student_id, "MATH201")
            sms.enroll_student_in_course(student.student_id, "CS101")
            sms.enroll_student_in_course(student.student_id, "ENGL101")
        else:
            sms.enroll_student_in_course(student.student_id, "ENGL101")
            sms.enroll_student_in_course(student.student_id, "MATH101")
    
    # Assign some grades
    for student in students:
        if "CS101" in student.enrolled_courses:
            sms.assign_grade(student.student_id, "CS101", "B+", "Fall", 2024)
        if "MATH101" in student.enrolled_courses:
            sms.assign_grade(student.student_id, "MATH101", "A-", "Fall", 2024)
        if "ENGL101" in student.enrolled_courses:
            sms.assign_grade(student.student_id, "ENGL101", "B", "Fall", 2024)
    
    print("Sample data loaded successfully!")
    
    # Run the interface
    interface = SMSInterface(sms)
    interface.run()


if __name__ == "__main__":
    run_demo()