from fastapi import APIRouter, HTTPException, Depends, status , Request
from sqlalchemy.orm import Session
from typing import List
from app.database import get_db
from app.models.simple_model import (
    SimpleModelCreateSchema,
    SimpleModelResponseSchema,
    SimpleModelUpdateSchema,
)
from app.services.simple_model_service import SimpleModelService
from app.global_constants import GlobalConstants
from app.auth import validate_and_authorize , AuthRoutes



class SimpleModelRoutes:
    router = APIRouter()

    """
    Creates a new SimpleModel entry.
    """
    @staticmethod
    @router.post("/simple-models", response_model=SimpleModelResponseSchema)
    def create_simple_model(simple_model: SimpleModelCreateSchema, db: Session = Depends(get_db)):
        try:
            service = SimpleModelService(db)
            return service.create_simple_model(simple_model)
        except Exception as e:
            print(f"{GlobalConstants.api_response_messages.error_in} create_simple_model route: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.internal_server_error,
                result=str(e),
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR
            )

    """
    Retrieves all SimpleModel entries.
    """
    @staticmethod
    @router.get("/simple-models", response_model=List[SimpleModelResponseSchema])
    @validate_and_authorize()
    def get_all_simple_models(request : Request, db: Session = Depends(get_db),token: dict = Depends(AuthRoutes.oauth2_scheme)):
        try:
            service = SimpleModelService(db)
            user_info = request.state.user_info
            print("User Info: ", user_info)
            return service.get_all_simple_models()
        except Exception as e:
            print(f"{GlobalConstants.api_response_messages.error_in} get_all_simple_models route: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.internal_server_error,
                result=str(e),
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR
            )

    """
    Retrieves a specific SimpleModel by ID.
    """
    @staticmethod
    @router.get("/simple-model/{simple_model_id}", response_model=SimpleModelResponseSchema)
    def get_simple_model_by_id(simple_model_id: int, db: Session = Depends(get_db)):
        try:
            service = SimpleModelService(db)
            return service.get_simple_model_by_id(simple_model_id)
        except Exception as e:
            print(f"{GlobalConstants.api_response_messages.error_in} get_simple_model_by_id route: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.internal_server_error,
                result=str(e),
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR
            )

    """
    Updates a SimpleModel entry by ID.
    """
    @staticmethod
    @router.patch("/simple-model", response_model=SimpleModelResponseSchema)
    def update_simple_model(payload: SimpleModelUpdateSchema, db: Session = Depends(get_db)):
        try:
            service = SimpleModelService(db)
            return service.update_simple_model(payload)
        except Exception as e:
            print(f"{GlobalConstants.api_response_messages.error_in} update_simple_model route: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.internal_server_error,
                result=str(e),
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR
            )

    """
    Deletes a SimpleModel entry by ID.
    """
    @staticmethod
    @router.delete("/simple-model/{simple_model_id}")
    def delete_simple_model(simple_model_id: int, db: Session = Depends(get_db)):
        try:
            service = SimpleModelService(db)
            return service.delete_simple_model(simple_model_id)
        except Exception as e:
            print(f"{GlobalConstants.api_response_messages.error_in} delete_simple_model route: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.internal_server_error,
                result=str(e),
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR
            )
