from fastapi import APIRouter, HTTPException, Depends, status, Request
from sqlalchemy.orm import Session
from typing import List
from uuid import UUID
from app.database import get_db
from app.models.todo_model import (
    ToDoCreateSchema,
    ToDoResponseSchema,
    ToDoUpdateSchema,
)
from app.services.todo_service import ToDoService
from app.global_constants import GlobalConstants


class ToDoRoutes:
    router = APIRouter()

    """
    Creates a new ToDo entry.
    """
    @staticmethod
    @router.post("/todos", response_model=ToDoResponseSchema)
    def create_todo(todo: ToDoCreateSchema, db: Session = Depends(get_db)):
        try:
            service = ToDoService(db)
            return service.create_todo(todo)
        except Exception as e:
            print(f"{GlobalConstants.api_response_messages.error_in} create_todo route: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.internal_server_error,
                result=str(e),
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            )

    """
    Retrieves all ToDo entries.
    """
    @staticmethod
    @router.get("/todos", response_model=List[ToDoResponseSchema])
    def get_all_todos(
        request: Request,
        db: Session = Depends(get_db)
    ):
        try:
            service = ToDoService(db)
            return service.get_all_todos()
        except Exception as e:
            print(f"{GlobalConstants.api_response_messages.error_in} get_all_todos route: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.internal_server_error,
                result=str(e),
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            )

    """
    Retrieves a specific ToDo by ID.
    """
    @staticmethod
    @router.get("/todo/{todo_id}", response_model=ToDoResponseSchema)
    def get_todo_by_id(todo_id: UUID, db: Session = Depends(get_db)):
        try:
            service = ToDoService(db)
            return service.get_todo_by_id(todo_id)
        except Exception as e:
            print(f"{GlobalConstants.api_response_messages.error_in} get_todo_by_id route: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.internal_server_error,
                result=str(e),
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            )

    """
    Updates a ToDo entry by ID.
    """
    @staticmethod
    @router.patch("/todo", response_model=ToDoResponseSchema)
    def update_todo(payload: ToDoUpdateSchema, db: Session = Depends(get_db)):
        try:
            service = ToDoService(db)
            return service.update_todo(payload)
        except Exception as e:
            print(f"{GlobalConstants.api_response_messages.error_in} update_todo route: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.internal_server_error,
                result=str(e),
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            )

    """
    Deletes a ToDo entry by ID.
    """
    @staticmethod
    @router.delete("/todo/{todo_id}")
    def delete_todo(todo_id: int, db: Session = Depends(get_db)):
        try:
            service = ToDoService(db)
            return service.delete_todo(todo_id)
        except Exception as e:
            print(f"{GlobalConstants.api_response_messages.error_in} delete_todo route: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.internal_server_error,
                result=str(e),
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            )



    """
    Retrieves all ToDo entries created by a specific user.
    """
    @staticmethod
    @router.get("/todos/created_by/{created_by_id}", response_model=List[ToDoResponseSchema])
    def get_todos_by_created_by_id(created_by_id: UUID, db: Session = Depends(get_db)):
        try:
            service = ToDoService(db)
            return service.get_todos_by_created_by_id(created_by_id)
        except Exception as e:
            print(f"{GlobalConstants.api_response_messages.error_in} get_todos_by_created_by_id route: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.internal_server_error,
                result=str(e),
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            )