from fastapi import APIRouter, HTTPException, Depends, status, Request
from sqlalchemy.orm import Session
from typing import List
from uuid import UUID
from app.database import get_db
from app.models.user_model import (
    User,
    UserCreateSchema,
    UserResponseSchema,
    UserUpdateSchema,
)
from app.services.user_service import UserService
from app.global_constants import GlobalConstants
from app.auth import validate_and_authorize, AuthRoutes
from app.Oauth import get_current_user

class UserRoutes:
    router = APIRouter()

    @staticmethod
    @router.post("/users", response_model=UserResponseSchema)
    def create_user(user: UserCreateSchema, db: Session = Depends(get_db)):
        try:
            service = UserService(db)
            return service.create_user(user)
        except Exception as e:
            print(f"{GlobalConstants.api_response_messages.error_in} create_user route: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.internal_server_error,
                result=str(e),
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            )

    @staticmethod
    @router.get("/users", response_model=List[UserResponseSchema])
    def get_all_users(
        request: Request,
        db: Session = Depends(get_db),
        current_user: User = Depends(get_current_user)
    ):
        try:
            service = UserService(db)
            return service.get_all_users()
        except Exception as e:
            print(f"{GlobalConstants.api_response_messages.error_in} get_all_users route: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.internal_server_error,
                result=str(e),
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            )

    @staticmethod
    @router.get("/user/{user_id}", response_model=UserResponseSchema)
    def get_user_by_id(user_id: UUID, db: Session = Depends(get_db)):
        try:
            service = UserService(db)
            return service.get_user_by_id(user_id)
        except Exception as e:
            print(f"{GlobalConstants.api_response_messages.error_in} get_user_by_id route: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.internal_server_error,
                result=str(e),
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            )

    @staticmethod
    @router.patch("/user", response_model=UserResponseSchema)
    def update_user(payload: UserUpdateSchema, db: Session = Depends(get_db)):
        try:
            service = UserService(db)
            return service.update_user(payload)
        except Exception as e:
            print(f"{GlobalConstants.api_response_messages.error_in} update_user route: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.internal_server_error,
                result=str(e),
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            )

    @staticmethod
    @router.delete("/user/{user_id}")
    def delete_user(user_id: UUID, db: Session = Depends(get_db)):
        try:
            service = UserService(db)
            return service.delete_user(user_id)
        except Exception as e:
            print(f"{GlobalConstants.api_response_messages.error_in} delete_user route: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.internal_server_error,
                result=str(e),
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            )
