from sqlalchemy.orm import Session
from sqlalchemy.exc import SQLAlchemyError
from app.models.simple_model import (
    SimpleModel,
    SimpleModelCreateSchema,
    SimpleModelUpdateSchema,
    SimpleModelResponseSchema,
)
from fastapi import status
from datetime import datetime, timezone
from app.global_constants import GlobalConstants
from sqlalchemy.future import select

class SimpleModelService:
    def __init__(self, db_session: Session):
        self.db_session = db_session


    def create_simple_model(self, simple_model_data: SimpleModelCreateSchema):
        try:
            new_simple_model = SimpleModel(
                name=simple_model_data.name,
                created_at=datetime.now(timezone.utc),
                updated_at=datetime.now(timezone.utc),
                created_by=simple_model_data.created_by,
                updated_by=simple_model_data.created_by,
            )
            self.db_session.add(new_simple_model)
            self.db_session.commit()
            self.db_session.refresh(new_simple_model)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.accepted,
                result=[SimpleModelResponseSchema(
                    id=new_simple_model.id,
                    name=new_simple_model.name,
                    created_at=new_simple_model.created_at.isoformat() if new_simple_model.created_at else None,
                    updated_at=new_simple_model.updated_at.isoformat() if new_simple_model.updated_at else None,
                    created_by=new_simple_model.created_by,
                    updated_by=new_simple_model.updated_by,
                )],
                status_code=status.HTTP_201_CREATED
            )
        except SQLAlchemyError as e:
            self.db_session.rollback()
            print(f"{GlobalConstants.api_response_messages.error_in} create_simple_model service: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.db_error,
                result=None,
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR
            )

    def get_all_simple_models(self):
        try:
            result = self.db_session.execute(select(SimpleModel))
            simple_models = result.scalars().all()
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.success,
                result=[SimpleModelResponseSchema(
                    id=model.id,
                    name=model.name,
                    created_at=model.created_at.isoformat() if model.created_at else None,
                    updated_at=model.updated_at.isoformat() if model.updated_at else None,
                    created_by=model.created_by,
                    updated_by=model.updated_by,
                ) for model in simple_models],
                status_code=status.HTTP_200_OK
            )
        except SQLAlchemyError as e:
            print(f"{GlobalConstants.api_response_messages.error_in} get_all_simple_models service: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.db_error,
                result=None,
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR
            )


    def get_simple_model_by_id(self, model_id: int):
        try:
            result = self.db_session.execute(
                select(SimpleModel).where(SimpleModel.id == model_id)
            )
            model = result.scalars().first()
            if not model:
                return GlobalConstants.return_api_response(
                    message=f"SimpleModel with ID {model_id} not found.",
                    result=[],
                    status_code=status.HTTP_400_BAD_REQUEST
                )
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.success,
                result=[SimpleModelResponseSchema(
                    id=model.id,
                    name=model.name,
                    created_at=model.created_at.isoformat() if model.created_at else None,
                    updated_at=model.updated_at.isoformat() if model.updated_at else None,
                    created_by=model.created_by,
                    updated_by=model.updated_by,
                )],
                status_code=status.HTTP_200_OK
            )
        except SQLAlchemyError as e:
            print(f"{GlobalConstants.api_response_messages.error_in} get_simple_model_by_id service: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.db_error,
                result=None,
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR
            )

    def update_simple_model(self, simple_model_update: SimpleModelUpdateSchema):
        try:
            result = self.db_session.execute(
                select(SimpleModel).where(SimpleModel.id == simple_model_update.simple_model_id)
            )
            model = result.scalars().first()
            if not model:
                return GlobalConstants.return_api_response(
                    message=f"SimpleModel with ID {simple_model_update.simple_model_id} not found.",
                    result=[],
                    status_code=status.HTTP_400_BAD_REQUEST
                )
            if simple_model_update.name is not None:
                model.name = simple_model_update.name
            if simple_model_update.updated_by is not None:
                model.updated_by = simple_model_update.updated_by
            model.updated_at = datetime.now(timezone.utc)
            self.db_session.commit()
            self.db_session.refresh(model)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.success,
                result=[SimpleModelResponseSchema(
                    id=model.id,
                    name=model.name,
                    created_at=model.created_at.isoformat() if model.created_at else None,
                    updated_at=model.updated_at.isoformat() if model.updated_at else None,
                    created_by=model.created_by,
                    updated_by=model.updated_by,
                )],
                status_code=status.HTTP_200_OK
            )
        except SQLAlchemyError as e:
            self.db_session.rollback()
            print(f"{GlobalConstants.api_response_messages.error_in} update_simple_model service: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.db_error,
                result=None,
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR
            )

    def delete_simple_model(self, model_id: int):
        try:
            result = self.db_session.execute(
                select(SimpleModel).where(SimpleModel.id == model_id)
            )
            model = result.scalars().first()
            if not model:
                return GlobalConstants.return_api_response(
                    message=f"SimpleModel with ID {model_id} not found.",
                    result=[],
                    status_code=status.HTTP_400_BAD_REQUEST
                )
            self.db_session.delete(model)
            self.db_session.commit()
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.success,
                result=None,
                status_code=status.HTTP_200_OK
            )
        except SQLAlchemyError as e:
            self.db_session.rollback()
            print(f"{GlobalConstants.api_response_messages.error_in} delete_simple_model service: ", e)
            return GlobalConstants.return_api_response(
                message=GlobalConstants.api_response_messages.db_error,
                result=None,
                status_code=status.HTTP_500_INTERNAL_SERVER_ERROR
            )