import streamlit as st
import yfinance as yf
import pandas as pd
import re
from collections import Counter

# --- Frequency-based text summarizer ---


def summarize_text(text, max_sentences=3):
    stopwords = set([
        "the", "is", "in", "and", "to", "of", "for", "a", "an", "on",
        "this", "that", "it", "as", "at", "be"
    ])
    sentences = re.split(r'(?<=[.!?]) +', text.strip())
    if not sentences:
        return ""
    words = [w.lower() for w in re.findall(r"\w+", text)
             if w.lower() not in stopwords]
    if not words:
        return sentences[0]
    frequencies = Counter(words)
    rankings = []
    for idx, sentence in enumerate(sentences):
        word_list = [w.lower() for w in re.findall(r'\w+', sentence)]
        score = sum(frequencies.get(w, 0) for w in word_list)
        rankings.append((score, idx, sentence))
    top_sentences = sorted(rankings, reverse=True)[:max_sentences]
    top_sentences.sort(key=lambda x: x[1])  # Preserve original order
    return " ".join([s[2] for s in top_sentences])


# --- Streamlit Page Configuration ---
st.set_page_config(page_title="Stock Dashboard", layout="centered")
st.title("Stock Price Dashboard & Text Summarizer")

# --- Sidebar for user inputs ---
st.sidebar.header("Stock Configuration")
ticker = st.sidebar.text_input("Enter Stock Ticker", "AAPL")
period = st.sidebar.selectbox(
    "Select Period", ["1mo", "3mo", "6mo", "1y", "2y", "5y"], index=3)
interval = st.sidebar.selectbox(
    "Select Interval", ["1d", "1wk", "1mo"], index=0)

# --- Download stock data ---


@st.cache_data
def load_data(ticker, period, interval):
    try:
        return yf.download(ticker, period=period, interval=interval)
    except Exception:
        return pd.DataFrame()


data = load_data(ticker, period, interval)

# --- Show Stock Chart ---
st.subheader(f"Price Chart for: {ticker.upper()}")
if not data.empty and "Close" in data.columns:
    st.line_chart(data["Close"])
    latest_close = float(data["Close"].iloc[-1])
    high_value = float(data["High"].max())
    low_value = float(data["Low"].min())

    st.metric("Latest Close", f"${latest_close:.2f}")
    st.metric("High", f"${high_value:.2f}")
    st.metric("Low", f"${low_value:.2f}")
else:
    st.warning("No valid stock data available for this ticker or time range.")

# --- Text Summarization Section ---
st.subheader("Stock-related Text Summarization")
text_input = st.text_area("Enter news or analysis related to this stock:")
max_sentences = st.slider("Max Sentences in Summary",
                          min_value=1, max_value=5, value=3)

if st.button("Summarize"):
    if text_input.strip():
        summary = summarize_text(text_input, max_sentences)
        st.success("Summary:")
        st.write(summary)
    else:
        st.warning("Please enter some text to summarize.")
